/************************************************************************
 * NAME:	utility.c
 *
 * DESCR:	Utility functions for all file system and file format
 *		implementations.
 *
 * NOTES:	
 ************************************************************************/
#include <stdio.h>
#include "standard.h"

#define BASELEN	8
#define EXTLEN	3

/************************************************************************
 * NAME:	fs_filename_combine()
 *
 * DESCR:	Given the "standard" filename format for all of the
 *		implemented filesystems, convert it to lower-case dot
 *		separated form.
 *
 * ARGS:	"filename" is a buffer of 13 characters.
 *
 * RETURNS:	
 *
 * NOTES:	- the filename is '\0' terminated
 *		- the padding in the filename is either space or '\0'
 ************************************************************************/
fs_filename_combine(char *filename,
		    char *base,
		    char *ext)
{
    char 	*ptr = filename;
    int		 i;

    for (i=0; i < BASELEN; i++, ptr++) {
	if (base[i] == ' ' || base[i] == '\0')
	    break;
	*ptr = tolower(base[i]);
    }

    *ptr++ = '.';

    for (i=0; i < EXTLEN; i++, ptr++) {
	if (ext[i] == ' ' || ext[i] == '\0')
	    break;
	*ptr = tolower(ext[i]);
    }

    if (i == 0) {	/* no extention at all	*/
	ptr--;		/*   get rid of '.'	*/
    }

    *ptr = '\0';
}


/************************************************************************
 * NAME:	fs_filename_split()
 *
 * DESCR:	Converts the given dot-separated filename into the "split"
 *		format that is used on most filesystems.
 *
 * ARGS:	"null_padding" is TRUE if '\0' padding should be used,
 *		otherwise space padding will be used.
 *		"base" and "ext" are buffers of 8 and 3 characters, and
 *		will contain the split-out filename when this returns.
 *
 * RETURNS:	
 *
 * NOTES:	- this is fixed at 8 and 3 for base and ext because that
 *		  is the common format for all implemented file-systems.
 *		  If this expands in the future, then expand this thing.
 *		  No need to over-engineer.
 *		- the filename is converted to upper case characters.
 *		- this routine is built to get a "basename" of the
 *		  filename...not a full path.
 ************************************************************************/
void
fs_filename_split(char *filename,
		   char *base,
		   char *ext,
		   int   null_padding)
{
    int		i;
    char	*ptr = filename;

    for (i=0; *ptr && i < BASELEN; i++, ptr++) {
	if (*ptr == '.') {
	    break;
	}
	base[i] = toupper(*ptr);
    }

    /* fill in base */

    for ( ;i<BASELEN; i++)
	base[i] = null_padding?'\0':' ';

    /* for the null_padding option, the buffer should be 1 bigger */

    if (null_padding) {
	base[i] = '\0';
    }
	
    for ( ;*ptr && *ptr != '.'; ptr++)
	;			/* long name, truncate	*/

    if (*ptr)
	ptr++;	/* skip over '.' */

    for ( i=0; *ptr && i < EXTLEN; ptr++, i++) {
	ext[i] = toupper(*ptr);
    }

    for ( ;i<EXTLEN; i++)
	ext[i] = null_padding?'\0':' ';

    /* for the null_padding option, the buffer should be 1 bigger */

    if (null_padding) {
	ext[i] = '\0';
    }
}

/************************************************************************
 * NAME:	fs_basename()
 *
 * DESCR:	Calculates the basename of the give file.  This is used
 *		on the UNIX side
 *
 * ARGS:	
 *
 * RETURNS:	a pointer into the "pathname" that is the basename of
 *		the file.
 *
 * NOTES:
 ************************************************************************/
char *
fs_basename(char *pathname)
{
    char *ptr;

    for (ptr = pathname+strlen(pathname)-1; ptr >= pathname && *ptr != '/'; ptr--)
	;

    ptr++;

    return(ptr);
}
